/**************************************************************************\
*//*! \file ef_xen_util.c Wrapper around Xen APIs

Copyright 2006 Solarflare Communications Inc,
               9501 Jeronimo Road, Suite 250,
               Irvine, CA 92618, USA

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License version 2 as published by the Free
Software Foundation, incorporated herein by reference.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

 *//*
\**************************************************************************/

#include <linux/if_ether.h>
#include <asm/io.h>
#include <asm/pgtable.h>
#include <asm/hypercall.h>
#include <xen/xenbus.h>
#include <xen/driver_util.h>
#include "ci/driver/efab/open.h"
#include "ci/xen/ef_xen_util.h"

/* This is only callable from Dom0 */
/* FIXME: Split this out into a privileged ops file. */
int ef_hyperop_pass_io_page(struct xenbus_device *dev, const char *name, 
                            unsigned long physpage, int domain, int rw)
{
  struct xenbus_transaction tr;
  dom0_op_t give_perm;
  int err = xenbus_transaction_start(&tr);
  err =  xenbus_printf(tr, dev->nodename, name, "%lx",physpage );
  xenbus_transaction_end(tr, err != 0 );
  if (err != 0) {
    printk("BEND: failed to write node '%s' err %d\n", name, err);
    return err;
  }
  /* Let the domain get at the H/W  */
  give_perm.cmd =  DOM0_IOMEM_PERMISSION;
  give_perm.u.iomem_permission.domain = domain;
  give_perm.u.iomem_permission.first_mfn = physpage;
  give_perm.u.iomem_permission.nr_mfns = 1;
  give_perm.u.iomem_permission.allow_access = 1;
  give_perm.u.iomem_permission.rw = rw;
  err = HYPERVISOR_dom0_op(&give_perm);
  if (err)
    printk("BEND: DOM0_IOMEM_PERMISSION returned %d\n", err);
  return err;
}

/* This is callable from anywhere */
int ef_hyperop_pass_mem_page(struct xenbus_device *dev, const char *name,
                             unsigned physpage, int domain)
{
  unsigned mfn =   mfn = pfn_to_mfn(physpage);
#if defined(VNIC_VERBOSE) || defined(BEND_VERBOSE)
  ci_log("allowing pfn %x to be mapped for '%s'", pfn, name); 
#endif
  return xenbus_grant_ring(dev, mfn);
}

struct ef_hyperop_mapping {
  struct vm_struct *vm;
  int pages;
  grant_handle_t grant_handles[0];
};

/* Map a series of grants into a contiguous virtual area */
void *ef_map_grants_contig(struct xenbus_device *dev, unsigned *grants, int npages, 
                           struct ef_hyperop_mapping **priv)
{
  struct ef_hyperop_mapping *map;
  struct vm_struct *vm;
  void *addr;
  int i;
  ci_log("Mapping %d consecutive grants. Ptr is %p", npages, grants);
  vm  = alloc_vm_area(PAGE_SIZE * npages);
  if (vm == NULL) {
    ci_log("Sorely disappointed by alloc_vm_area.");
    return NULL;
  }
  /* Get a structure in which we will record all the info needed to undo
  * the mapping. */
  map = kzalloc(sizeof(struct ef_hyperop_mapping)  + 
                npages * sizeof(grant_handle_t), GFP_KERNEL);
  if (map == NULL) {
    ci_log("No luck allocating ef_hyperop_mapping");
    free_vm_area(vm);
  }
  map->vm = vm;
  map->pages = npages;

  if (priv)
    *priv = map;
  else
    ci_log("Warning: will not be able to undo mapping as no priv pointer given");

  /* Do the actual mapping */
  addr = vm->addr;
  ci_log("addr is %p", addr);
  for (i = 0; i < npages; i++) {
    int rc;
    rc = xenbus_map_ring(dev, grants[i], map->grant_handles + i, addr);
    if (rc != 0) {
      ci_log("Wailing and gnashing teeth, but otherwise no idea what to do. %d\n", rc);
    }
    addr = (void*)((unsigned long)addr + PAGE_SIZE);
  }
  ci_log("vnic_map_grants done.\n");
  return vm->addr;
}

/* Undo the result of the mapping */
void ef_hyperop_unmap_contig(struct xenbus_device *dev,
                             struct ef_hyperop_mapping *priv)
{
  struct ef_hyperop_mapping *map = (struct ef_hyperop_mapping *)priv;
  void *addr = map->vm->addr;
  int npages = map->pages;
  int i;
  ci_log("Unmapping %d pages starting at %p", npages, addr);
  for (i = 0; i < npages; i++) {
    xenbus_unmap_ring(dev, map->grant_handles[i], addr);
    addr = (void*)((unsigned long)addr + PAGE_SIZE);
  }
  free_vm_area(map->vm);
  kfree(map);
}


/* This used to be accessible, now they declare it static, so I've
   copied for now.  Should disappear when we merge into xen tree */
int xen_net_read_mac(struct xenbus_device *dev, u8 mac[])
{
  char *s, *e, *macstr;
  int i;

  macstr = s = xenbus_read(XBT_NIL, dev->nodename, "mac", NULL);
  if (IS_ERR(macstr))
    return PTR_ERR(macstr);

  for (i = 0; i < ETH_ALEN; i++) {
    mac[i] = simple_strtoul(s, &e, 16);
    if ((s == e) || (*e != ((i == ETH_ALEN-1) ? '\0' : ':'))) {
      kfree(macstr);
      return -ENOENT;
    }
    s = e+1;
  }

  kfree(macstr);
  return 0;
}

