/**************************************************************************\
*//*! \file vnic.h Structures and function definitions for VNIC driver

Copyright 2006 Solarflare Communications Inc,
               9501 Jeronimo Road, Suite 250,
               Irvine, CA 92618, USA

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License version 2 as published by the Free
Software Foundation, incorporated herein by reference.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

 *//*
\**************************************************************************/

#ifndef VNIC_H
#define VNIC_H

#include <ci/tools.h>
#include <etherfabric/vi.h>
/* FIXME: Xen-ism */
#include <ci/xen/ef_shared_fifo.h>
#include <ci/xen/ef_msg_iface.h>
#include <ci/xen/ef_cuckoo_hash.h>

#if defined(VNIC_VERBOSE)
#define VNIC_VERB(_x) _x
#else
#define VNIC_VERB(_x) do {} while(0)
#endif

#define EF_RX_ID_FIFO_SIZE              512 
#define EF_TX_ID_FIFO_SIZE              1024

#define EF_VNIC_RX_DESC_BATCH		16

/*! \file vnic.h Front end driver
 *
 * The guest driver / front end is split into 3 parts.
 *   BEC  - The back end communications layer.
 *   VNIC - The guest OS net driver.
 *   SVH  - The Semi-virtual hardware layer.
 * 
 * BEC is responsible for resource management, interrupts and device
 * status.  VNIC provides the guest OS NIC driver.  SVH processes
 * packets and passes them to/from the VI or SHM buffer.
 * 
 * There is a universal structure which holds data for all three
 * components.  This is good enough to start with, but might change in
 * the future so we should have 3 types of pointer and macros to map
 * between them.
 * 
 * The following functions are defined.  They all expect to be called
 * from a non-blocking context.  All virtual devices are treated as
 * independent, so there are no concurrency guarantees between them.
 * Global entry points are in BEC for interrupts from the back end and
 * in VNIC for guest OS communications.
 */

typedef ci_uint32 ef_vnic_size_t;
typedef struct ef_vnic_svh_s ef_vnic_svh;
typedef struct ef_vnic_bec_s ef_vnic_bec;
typedef struct ef_vnic_osd_s ef_vnic_osd;

typedef struct sk_buff ef_vnic_pktbuff;

enum ef_vnic_post_status {
  EF_VNIC_STATUS_GOOD,
  EF_VNIC_STATUS_BUSY,
  EF_VNIC_STATUS_FAIL
};

typedef struct {
  /* Software resources. */
  void *rx_packets;
  void *tx_packets;
  struct ef_shared_page *shared_page;

  /* Hardware resources. */
  void *tx_dmaq;
  void *tx_doorbell;
  void *rx_dmaq;
  void *rx_doorbell;
  void *evq_base;
  int evq_order;
  void *timer_reg;
} ef_vnic_resources;

/** Functions for accessing the virtualized hardware. */
typedef struct {
  int (*ctor)(ef_vnic_svh *svh, struct ef_msg_hw *hw_msg);
  void (*dtor)(ef_vnic_svh *svh);
  void (*add_bufs)(ef_vnic_svh *svh, struct ef_msg *msg);
  /** Send a packet.  See ef_vnic_svh_tx_post. */
  enum ef_vnic_post_status (*tx_post)(ef_vnic_svh *svh,
                                      ef_vnic_pktbuff *pktbuff);
  /** Release all packet buffers.  See ef_vnic_svh_unpost_all. */
  void (*unpost_all)(ef_vnic_svh *svh);
  /** Service the virtualized hardware.  See ef_vnic_svh_poll. */
  int  (*poll)(ef_vnic_svh *svh,
               int rx_packets);
  /** Update the virtualized hardware following a MTU update.  See
   * ef_vnic_svh_updated_mtu. */
  void (*updated_mtu)(ef_vnic_svh *svh);
  /** Check for interrupts only.  See
   * ef_vnic_svh_check_interrupts. */
  ci_boolean_t (*check_interrupts)(ef_vnic_svh *svh);
  /** Check for and enable interrupts.  See
   * ef_vnic_svh_enable_interrupts. */
  ci_boolean_t (*enable_interrupts)(ef_vnic_svh *svh);
  /** Disable interrupts.  See ef_vnic_svh_disable_interrupts. */
  void         (*disable_interrupts)(ef_vnic_svh *svh);
} ef_vnic_svh_functions;

typedef struct {
  /** True if the link is up. */
  ci_boolean_t is_up;
  /* Speed? */
  /* Duplex? */

  /** The link MTU. */
  ef_vnic_size_t mtu;

  /** The required size for RX buffers in bytes. */
  ef_vnic_size_t rx_buffer_length;
} ef_vnic_link_info;

/** State of the SVH.  Only ever instantiated in ef_vnic_all. */
struct ef_vnic_svh_s {
  ef_vnic_svh_functions *hw_fns;

  struct ef_vnic_bufinfo *bufs;

  /* Fastpath stats. */
  int fastpath_rx_pkts;
  int fastpath_rx_bytes;
  int fastpath_tx_pkts; 
  int fastpath_tx_bytes;

  /* Fast path events */
  int fastpath_tx_busy;

  /* TX DMA queue status */
  int fastpath_tx_completions;
  int fastpath_tx_pending_max;

  /* Hardware */
  ef_vi vi;
  ef_vi_state_pub ep_state;
  efab_dma_q_state txqs;
  efab_dma_q_state rxqs;
  ef_eventq evq;
  ef_eventq_state evq_state;
  void *evq_timer_page;
  void *evq_ptr_page;
  ci_uint32 *evq_ptr;
  ci_uint8 phys_port;

  /* Event queue pages exported to the back end. */
  struct ef_hyperop_mapping *evq_mapping;

  /* RX DMA queue status */
  ci_uint32 rx_dma_level;
  ci_uint32 rx_dma_max;

  /** The number of events processed. */
  ci_uint32 event_count;

  /** The number of events since the last interrupts. */
  ci_uint32 event_count_since_irq;

  /** The number of events since the last interrupts. */
  ci_uint32 events_per_irq_max;

  /* Some RX descriptors waiting to be pushed to the card. */
  int rx_dma_batched;
  ef_vi_receive_desc rx_batch_descs[EF_VNIC_RX_DESC_BATCH];

  /* Hash table of local mac addresses to decide whether to try fast path */
  ef_cuckoo_hash_table local_macs_table;
  spinlock_t local_macs_lock;

  int rx_pkt_stride;
  int rx_skb_stride;

  /* Number of remaining buffers that have yet to be sent to dom0 for
     remapping - used to make ef_vnic_ef1_buffer_requests sort-of
     re-entrant */
  ci_uint32 buffer_requests_left;
};

/** State of the OSD.  Only ever instantiated in ef_vnic_all. */
struct ef_vnic_osd_s {
  struct net_device *net_dev;

  /** A set of bits, each indicating a reason that packet processing
   * should be halted. */
  unsigned int stop_reasons;
  ci_boolean_t irq_enabled;
  spinlock_t stop_lock;

  /** A spare slot for a TX packet.  This is treated as an extension
   * of the DMA queue. */
  struct sk_buff *tx_skb;

  /** True if there is a MTU change outstanding. */
  ci_boolean_t mtu_changed;

  /** Mutex to ensure that only thread is filling the skb pool. */
  struct semaphore rx_fill_mutex;

  /** The maximum number of rx buffers to be posted.  This is a
   * configuration parameter. */
  ci_uint32 rx_max_fill;

  /** The minimum fill level of rx buffers which has been observed. */
  ci_uint32 rx_min_fill;

  /** The number of rx allocs on the fast path. */
  ci_uint32 fastpath_rx_alloc;

  /** The number of slow rx refills. */
  ci_uint32 rx_refills;

  /** The number of interrupts. */
  ci_uint32 irq_count;

  /** The number of useless interrupts. */
  ci_uint32 useless_irq_count;

  /** The number of polls scheduled. */
  ci_uint32 poll_schedule_count;

  /** The number of polls called. */
  ci_uint32 poll_call_count;

  /** The number of rechecks. */
  ci_uint32 poll_reschedule_count;

  /** A list of pre-allocated skb structures. */
  struct sk_buff_head rx_skb_list;

  /** Scheduled when rx buffers need to be posted. */
  struct work_struct rx_refill_work;

  /** Raised when all buffers have been unposted. */
  struct semaphore unpost_sem;

  /** Scheduled when the VI needs resetting. */
  struct work_struct reset_work;

  /** Mutex to protect VI resets. */
  struct semaphore reset_sem;

  /** Network statistics. */
  struct net_device_stats stats;
};

/** State of the BEC.  Only ever instantiated in ef_vnic_all. */
struct ef_vnic_bec_s {
  struct xenbus_device *dev;
  ef_notify_t channel;     /* Event channel for messages */
  ef_notify_t net_channel; /* Event channel for network activity. */
  struct ef_shared_page *shared_page;
  struct work_struct state_change;
  int next_state;

  /* Message Qs, 1 each way. */
  sh_msg_fifo2 to_dom0;
  sh_msg_fifo2 from_dom0;
  /* Byte FIFO containing slow path Rx packets. */
  sh_byte_fifo2 rcv_pkts;
  /* Byte FIFO containing slow path Tx packets. */
  sh_byte_fifo2 snd_pkts;
  /* Infrastructure for setting up the FIFO */
  struct page **rcv_page_list; /* List of pages */
  struct page **snd_page_list; /* List of pages */
  int rcv_pages; /* Number of pages. */
  int snd_pages; /* Number of pages. */
};


typedef struct {
  ef_vnic_link_info inf;
  ef_vnic_svh svh;
  ef_vnic_osd osd;
  ef_vnic_bec bec;
} ef_vnic_all;

#define EF_VNIC_SVH_FROM_OSD(p) (&CI_CONTAINER(ef_vnic_all, osd, (p))->svh)
#define EF_VNIC_SVH_FROM_BEC(p) (&CI_CONTAINER(ef_vnic_all, bec, (p))->svh)
#define EF_VNIC_OSD_FROM_SVH(p) (&CI_CONTAINER(ef_vnic_all, svh, (p))->osd)
#define EF_VNIC_OSD_FROM_BEC(p) (&CI_CONTAINER(ef_vnic_all, bec, (p))->osd)
#define EF_VNIC_BEC_FROM_OSD(p) (&CI_CONTAINER(ef_vnic_all, osd, (p))->bec)
#define EF_VNIC_BEC_FROM_SVH(p) (&CI_CONTAINER(ef_vnic_all, svh, (p))->bec)
#define EF_VNIC_INF_FROM_OSD(p) (&CI_CONTAINER(ef_vnic_all, osd, (p))->inf)
#define EF_VNIC_INF_FROM_SVH(p) (&CI_CONTAINER(ef_vnic_all, svh, (p))->inf)

extern ef_vnic_svh_functions ef_vnic_svh_ef1;
extern ef_vnic_svh_functions ef_vnic_svh_null;

extern
int ef_vnic_svh_ctor(ef_vnic_svh *svh, struct ef_msg_hw *hw_msg);
extern
void ef_vnic_svh_dtor(ef_vnic_svh *svh);

/**
 * Called by BEC when a VI has been found.
 *
 * @v   osd      The OSD structure to be initialised.
 * @ret success  TRUE if the operation succeeded.
 *
 * Requests that the OSD initialise the specified structure.  On
 * success, there will be a matching call to ef_vnic_osd_remove().  On
 * failure, no other BEC->VNIC calls will be made.  This is the first
 * call made for the virtual device and will not be concurrent with
 * any other BEC->VNIC call. */
extern
ci_boolean_t ef_vnic_osd_probe(ef_vnic_osd *osd);

/**
 * Called by BEC when a VI is being removed.
 *
 * @v   osd      The OSD structure to be destroyed.
 *
 * This is the final call for this virtual device and will not be
 * concurrent with any other BEC->VNIC call.  This call may block. */
extern
void ef_vnic_osd_remove(ef_vnic_osd *osd);

/**
 * Called by BEC to suspend OSD operations.
 *
 * @v   osd      The OSD instance to be suspended.
 *
 * The virtual resources are being removed temporarily.  The OSD
 * should respond by suspending processing, unpost all buffers and
 * disabling interrupts.  This will called alternately with
 * ef_vnic_osd_resume.  This call can block. */
extern
void ef_vnic_osd_suspend(ef_vnic_osd *osd);

/**
 * Called by BEC to resume OSD operations.
 *
 * @v   osd      The OSD instance to be resumed.
 *
 * This call informs the OSD that the virtual resources have been
 * replaced, so processing can continue.  This will be called
 * alternately with ef_vnic_osd_suspend. */
extern
void ef_vnic_osd_resume(ef_vnic_osd *osd);

/**
 * Called by BEC to  OSD that an interrupt has arrived.
 *
 * @v   osd      The OSD instance to process the interrupt.
 *
 * An interrupt has arrived which needs handling by calling
 * ef_vnic_svh_poll(). */
extern
void ef_vnic_osd_interrupt(ef_vnic_osd *osd);

/**
 * Called by BEC to handle a buffer response.
 *
 * @v   svh     The SVH instance to process the response.
 * @v   msg     The message to process.
 *
 * The buffers contained in the message are added to the SVH buffer
 * pool.
 */
extern
void ef_vnic_svh_add_bufs(ef_vnic_svh *svh, struct ef_msg *msg);

/**
 * Put a packet on the tx DMA queue.
 *
 * @v  svh         The SVH instance to accept the packet.
 * @v  pktbuff     An OSD packet buffer to send.
 *
 * This indicates to the SVH that the OSD wishes to send a packet.  On
 * success, the packet is owned by the SVH.
 */
extern
enum ef_vnic_post_status ef_vnic_svh_tx_post(ef_vnic_svh *svh,
                                             ef_vnic_pktbuff *pktbuff);

/** Request that all posted buffers be removed.
 *
 * @v  svh   The SVH instance to flush.
 *
 * The SVH will call ef_vnic_unposted_all when all buffers have been
 * unposted.  There must not be an outstanding unpost request.  This
 * will not be called concurrently with any of the descriptor pushing
 * calls. */
extern
void ef_vnic_svh_unpost_all(ef_vnic_svh *svh);

/**
 * Process events in response to an interrupt.
 *
 * @v   svh        The SVH instance to poll.
 * @v   rx_packets The maximum number of rx packets to process.
 * @ret rx_done    The number of rx packets processed.
 *
 * The SVH will process events until there are no more events
 * remaining or the specified number of rx packets has been processed.
 * The split from the BEC->VNIC interrupt call is to allow Linux NAPI
 * polling. */
extern
int ef_vnic_svh_poll(ef_vnic_svh *svh, int rx_packets);

/**
 * Get the location of the header data of a packet buffer.
 *
 * @v   pktbuff   The packet buffer to query.
 * @ret data      The header data.
 *
 * This function returns the address of the first fragment of a packet
 * buffer.  This will usually include the ethernet header.
 */
ci_inline
void *ef_vnic_pktbuff_get_header_data(ef_vnic_pktbuff *pktbuff)
{
  return pktbuff->head;
}

/**
 * Get the length of the header data of a packet buffer.
 *
 * @v   pktbuff   The packet buffer to query.
 * @ret length    The length of the header data.
 *
 * This function returns the length of the first fragment of a packet
 * buffer.  This will usually include the ethernet header.
 */
ci_inline
ef_vnic_size_t ef_vnic_pktbuff_get_header_length(ef_vnic_pktbuff *pktbuff)
{
  return skb_headlen(pktbuff);
}

/**
 * Get the total length of a packet buffer.
 *
 * @v   pktbuff   The packet buffer to query.
 * @ret length    The total length of the packet.
 *
 * This function returns the sum of the lengths of all the fragment of
 * a packet buffer.
 */
ci_inline
ef_vnic_size_t ef_vnic_pktbuff_get_total_length(ef_vnic_pktbuff *pktbuff)
{
  return pktbuff->len;
}

/**
 * Iterate over the fragments of a packet buffer.
 *
 * @v   pktbuff  The packet buffer to examine.
 * @v   idx      A variable name for the fragment index.
 * @v   data     A variable name for the address of the fragment data.
 * @v   length   A variable name for the fragment length.
 * @v   code     A section of code to execute for each fragment.
 *
 * This macro iterates over the fragments in a packet buffer and
 * executes the code for each of them.
 */
#define EF_VNIC_PKTBUFF_FOR_EACH_FRAGMENT(pktbuff, frag_idx,  \
                                          frag_data,          \
                                          frag_len,           \
                                          code)               \
  do {                                                        \
    int frag_idx;                                             \
    void *frag_data;                                          \
    ef_vnic_size_t frag_len;                                  \
                                                              \
    frag_data = pktbuff->data;                                \
    frag_len = skb_headlen(pktbuff);                          \
    frag_idx = 0;                                             \
    while(1) { /* For each fragment */                        \
      code;                                                   \
      if ( frag_idx >= skb_shinfo(pktbuff)->nr_frags ) {      \
        break;                                                \
      } else {                                                \
        skb_frag_t *fragment;                                 \
        fragment = &skb_shinfo(pktbuff)->frags[frag_idx];     \
        frag_len = fragment->size;                            \
        frag_data = ( (void*)page_address(fragment->page) +   \
                      fragment->page_offset );                \
      };                                                      \
      frag_idx++;                                                  \
    }                                                         \
  } while(0)

/**
 * Allocates a buffer for rx packets.
 *
 * @v   osd        The OSD instance being informed.
 * @v   len        The size of the buffer in bytes.
 * @ret pktbuff    A new OS packet buffer.
 *
 * Called by the SVH when it needs a packet buffer.  Returns a pointer
 * to a new packet buffer with a single fragment if the allocation
 * succeeded or NULL on failure.  The buffer is owned by the SVH on
 * return.
 */
extern
ef_vnic_pktbuff *ef_vnic_osd_new_rx_buffer(ef_vnic_osd *osd, int len);

/**
 * Indicates that a packet has been received.
 *
 * @v   osd        The OSD instance being informed.
 * @v   pktbuff    The packet buffer passed to ef_vnic_svh_rx_post.
 * @v   len        The number of bytes in the packet.
 *
 * Called from ef_vnic_svh_poll() after a call to ef_vnic_svh_rx_post
 * for this packet.  The buffer ownership is passed back to the
 * VNIC. */
extern
void ef_vnic_osd_rx_complete(ef_vnic_osd *osd,
                             ef_vnic_pktbuff *pktbuff,
                             ef_vnic_size_t len);

/**
 * Indicates that a packet has been transmitted.
 *
 * @v   osd        The OSD instance being informed.
 * @v   pktbuff    The packet buffer passed to ef_vnic_svh_tx_reserve.
 *
 * Called from ef_vnic_svh_poll() after a call to ef_vnic_svh_tx_push
 * for this packet.  The buffer ownership is passed back to the
 * VNIC. */
extern
void ef_vnic_osd_tx_complete(ef_vnic_osd *osd,
                             ef_vnic_pktbuff *pktbuff);

/**
 * Indicates that a packet buffer has been unposted.
 *
 * @v   osd        The OSD instance being informed.
 * @v   pktbuff    
 *                 The OS packet buffer passed to
 *                 ef_vnic_svh_tx_reserve or ef_vnic_svh_rx_post.
 * @v   is_tx      True if the packet is from the TX queue.
 *
 * Called from ef_vnic_svh_poll() following a call to
 * ef_vnic_svh_unpost_all or from ef_vnic_svh_updated_mtu() while this
 * packet was owned by the SVH.  The buffer ownership is passed back
 * to the VNIC. */
extern
void ef_vnic_osd_unposted(ef_vnic_osd *osd,
                          ef_vnic_pktbuff *pktbuff,
                          ci_boolean_t is_tx);

/**
 * Indicates that all packet buffers have been unposted.
 *
 * @v   osd        The OSD instance being informed.
 *
 * Called from ef_vnic_svh_poll() following a call to
 * ef_vnic_svh_unpost_all.  The OSD is now free to post more
 * buffers. */
extern
void ef_vnic_osd_unposted_all(ef_vnic_osd *osd);

/**
 * Indicates that the link status has changed.
 *
 * @v   osd        The OSD instance being informed.
 * @v   is_up      The new link status.
 *
 * Called from ef_vnic_svh_poll() to indicate that the link status has
 * changed. */
extern
void ef_vnic_osd_change_link_status(ef_vnic_osd *osd,
                                    ci_boolean_t is_up);

/**
 * Indicates that the MTU has changed.
 *
 * @v   osd        The OSD instance being informed.
 * @v   mtu        The link MTU.
 * @v   rx_size    The required rx buffer size.
 *
 * Called from ef_vnic_svh_poll() to indicate that the MTU has
 * changed.  The OSD is responsible for updating the link info and
 * calling ef_vnic_svh_updated_mtu. */
extern
void ef_vnic_osd_change_mtu(ef_vnic_osd *osd,
                            ef_vnic_size_t mtu,
                            ef_vnic_size_t rx_size);

/**
 * Indicates that an MTU change has been processed.
 *
 * @v   svh        The relevant SVH instance.
 *
 * Called by OSD to request that the RX buffer ring be updated as a
 * result of an MTU change.  This can result in RX buffers being
 * unposted.  It may not be called concurrently with ef_vnic_svh_poll,
 * ef_vnic_svh_rx_post or ef_vnic_svh_rx_push.
 */
extern
void ef_vnic_svh_updated_mtu(ef_vnic_svh *svh);

/**
 * Informs the back end that there is work to be done.
 *
 * @v   bec        The relevant BEC instance.
 * 
 * This is called by SVH to signal to the back end that there is work
 * for it to do. */
extern
void ef_vnic_bec_doorbell(ef_vnic_bec *bec);

/**
 * Called by OSD to check if there are any pending interrupts.
 *
 * @v   svh        The relevant SVH instance.
 *
 * Checks for pending interrupts without delivering them.
 */
extern
ci_boolean_t ef_vnic_svh_check_interrupts(ef_vnic_svh *svh);

/**
 * Called by OSD to enable the delivery of interrupts.
 *
 * @v   svh        The relevant SVH instance.
 *
 * Enables interrupts.  Outstanding interrupts may be delivered
 * immediately.
 */
extern
ci_boolean_t ef_vnic_svh_enable_interrupts(ef_vnic_svh *svh);

/** Called by OSD to inhibit delivery of interrupts.
 *
 * @v   svh        The relevant SVH instance.
 *
 * Any outstanding calls to ef_vnic_osd_interrupt will terminate
 * before this function returns.
 */
extern
void ef_vnic_svh_disable_interrupts(ef_vnic_svh *svh);

/** Called by OSD to inhibit delivery of interrupts.
 *
 * @v   bec        The relevant BEC instance.
 *
 * Any outstanding calls to ef_vnic_osd_interrupt will terminate
 * before this function returns.
 */
extern
void ef_vnic_bec_disable_interrupts(ef_vnic_bec *bec);

/**
 * Called by OSD to enable the delivery of interrupts.
 *
 * @v   bec        The relevant BEC instance.
 *
 * Enables interrupts.  Outstanding interrupts may be delivered
 * immediately.
 */
extern
void ef_vnic_bec_enable_interrupts(ef_vnic_bec *bec);

/**
 * Called by OSD to get the MAC address.
 *
 * @v   svh        The SVH instance being queried.
 * @v   buffer     A 6 byte buffer for the MAC address.
 *
 * Copies the MAC address into the buffer provided.
 */
extern
void ef_vnic_svh_get_mac_addr(ef_vnic_svh *svh,
                              ci_uint8 *buffer);

/**
 * Called by SVH to send a message
 *
 * @v   bec       The BEC that owns the message queue
 * @v   msg       The message
 */
ci_inline
int ef_bec_send_msg(ef_vnic_bec *bec, struct ef_msg *msg) {
  return ef_msg_send_notify(bec->shared_page, bec->channel,
                            &bec->to_dom0, msg);
}

/**
 * Called by SVH to get a lock on an in place message
 *
 * @v   bec       The BEC that owns the message queue
 * @v   flags     A place to store the lock flags
 * 
 * Returns a pointer to the message in place in the queue, or NULL on
 * error.  Queue is locked until message is completed or aborted.
 */
ci_inline
struct ef_msg *ef_bec_start_msg(ef_vnic_bec *bec, unsigned *flags) {
  return ef_msg_start_send(bec->shared_page, &bec->to_dom0, flags);
}

/**
 * Called by SVH to complete a message started with ef_bec_start_msg
 *
 * @v   bec       The BEC that owns the message queue
 * @v   flags     The lock flags returned by ef_bec_start_msg()
 * 
 */
ci_inline
void ef_bec_complete_msg(ef_vnic_bec *bec, unsigned *flags) {
  return ef_msg_complete_send_notify(bec->shared_page, &bec->to_dom0, flags,
                                     bec->channel);
}

/**
 * Called by SVH to abort a message started with ef_bec_start_msg
 *
 * @v   bec       The BEC that owns the message queue
 * @v   flags     The lock flags returned by ef_bec_start_msg()
 * 
 */
ci_inline
void ef_bec_abort_msg(ef_vnic_bec *bec, unsigned *flags) {
  return ef_msg_abort_send(bec->shared_page, &bec->to_dom0, flags);
}

#endif
/*
 * Local variables:
 *  c-basic-offset: 2
 *  c-indent-level: 2
 *  tab-width: 8
 * End:
 */
