import SymbolTable
import array

class DomainImage:
    '''
    Domain address space and symbols environment to analyze dump file
    '''

    def __init__(self, dump, vmlinux, domain_pt):
        self.dump = dump
        self.pt = domain_pt

        # symbol table
        self.st = SymbolTable.SymbolTable()
        self.st.loadSymbolFromObjectFile(vmlinux)

        ## DWARF debug information tree isn't used, yet
        #self.dwarf = debug_info.TagTree()
        #self.dwarf.buildTree(vmlinux)

        self.max_pfn = self.get_max_pfn()
        self.pfn2mfn = self.get_p2m()

    def get_max_pfn(self):
        '''return max page frame number'''

        # read data
        return self.dump.read_struct(self.pt.v2m(self.st['max_pfn']), 'L')[0]

    def get_p2m(self):
        '''return pfn2mfn mapping dict'''

        p2m_table = self.dump.read_struct(self.pt.v2m(self.st['phys_to_machine_mapping']), 'L')[0]

        # make pfn2mfn from phys_to_machine_mapping
        # XXX x86_32 specific
        p2m = array.array('L')
        p2m.fromstring(self.pt.load_data(p2m_table, self.max_pfn * 4))
        pfn2mfn = {}
        for pfn, mfn in enumerate(p2m):
            if not mfn & 0x80000000L:
                pfn2mfn[pfn] = mfn
        return pfn2mfn
