#!/usr/bin/env python

'''Convert register information between ELF and Xen '''
import struct
import re

class BaseRegister(dict):
    '''Register abstruction class'''

    def __init__(self):
        self.elffmt = ''
        self.elfregs = ()
        self.xenfmt = ''
        self.xenregs = ()

    def __repr__(self):
        ret = ['{']
        keys = self.keys()
        keys.sort()
        for key in keys:
            ret.append(key)
            ret.append(': %08x, ' % self[key])
        ret.append('}')
        return ''.join(ret)
        
    def fromString(self, fmt, names, regstring):
        '''convert from string'''

        regs = struct.unpack(fmt, regstring)
        for key, val in zip(names, regs):
            self[key] = val
        return self

    def fromElf(self, regstring):
        '''convert from ELF context format'''
        
        return self.fromString(self.elffmt, self.elfregs, regstring)

    def fromXen(self, regstring):
        '''convert from Xen context format'''
        
        return self.fromString(self.xenfmt, self.xenregs, regstring)

    def asString(self, fmt, names):
        '''convert to string'''

        regs = []
        for regname in names:
            if regname in self:
                regs.append(self[regname])
            else:
                regs.append(0)
        return struct.pack(fmt, *regs)

    def asElf(self):
        '''convert to ELF context format'''

        return self.asString(self.elffmt, self.elfregs)
        
    def asXen(self):
        '''convert to Xen context format'''

        return self.asString(self.xenfmt, self.xenregs)



class Register_x86_32(BaseRegister):
    '''xs86_32 register info'''

    def __init__(self):
        BaseRegister.__init__(self)

        # from: $(LINUX)/include/asm-i486/user.h
        self.elffmt = 'LLLLLLLHHHHHHHHLLHHLLHH'
        self.elfregs = ('ebx', 'ecx', 'edx', 'esi', 'edi', 'ebp', 'eax',
                        'ds', '__ds', 'es', '__es',
                        'fs', '__fs', 'gs', '__gs',
                        'orig_eax', 'eip',
                        'cs', '__cs',
                        'eflags', 'esp',
                        'ss', '__ss') 

        # from: $(XEN)/xen/include/public/arch-x86_32.h
        self.xenfmt = 'LLLLLLLHHLHBBLLHHHHHHHHHH'
        self.xenregs = (
            'ebx', 'ecx', 'edx', 'esi', 'edi', 'ebp', 'eax',
            'error_code', 'entry_vector', 'eip', 'cs',
            'saved_upcall_mask', '_pad0',
            'eflags',
            'esp',
            'ss', '_pad1', 'es', '_pad2',
            'ds', '_pad3', 'fs', '_pad4',
            'gs', '_pad5',
            )


class Register_x86_64(BaseRegister):
    '''xs86_64 register info'''
    # XXX not used or tested yet

    def __init__(self):
        BaseRegister.__init__(self)

        # from: $(LINUX)/include/asm-x86_64/user.h
        self.elffmt = 'QQQQQQQQQQQQQQQQQQQQQQQQQQQ'
        self.elfregs = (
            'r15', 'r14', 'r13', 'r12', 
            'rbp', 'rbx', 'r11', 'r10', 
            'r9', 'r8', 'rax', 'rcx', 
            'rdx', 'rsi', 'rdi', 'orig_rax', 
            'rip', 'cs', 'eflags', 
            'rsp', 'ss', 
            'fs_base', 'gs_base', 
            'ds', 'es', 'fs', 'gs') 

        # from: $(XEN)/xen/include/public/arch-x86_64.h
        self.xenfmt = 'QQQQQQQQQQQQQQQQQQQLLQHHBBBBQQHHHHHHHHHHHHHHHHHHHH'
        self.xenregs = (
            'r15', 'r14', 'r13', 'r12',
            'rbp', 'rbx', 'r11', 'r10',
            'r9', 'r8', 'rax', 'rcx',
            'rdx', 'rsi', 'rdi',
            'error_code',
            'entry_vector',
            'rip',
            'cs', '_pad0',
            'saved_upcall_mask', '_pad10', '_pad11', '_pad12',
            'rflags',
            'rsp',
            'ss', '_pad20', '_pad21', '_pad22',
            'es', '_pad30', '_pad31', '_pad32',
            'ds', '_pad40', '_pad41', '_pad42',
            'fs', '_pad50', '_pad51', '_pad52',
            'gs', '_pad60', '_pad61', '_pad62',
            )

class Register_ia64(BaseRegister):
    '''IA64 register info'''
    
    # XXX not used or tested yet
    def __init__(self):
        BaseRegister.__init__(self)

        # from: $(LINUX)/include/asm-ia64/ptrace.h
        self.elffmt = '44Q6Q'
        self.elfregs = (
            'b6', 'b7', 'ar_csd', 'ar_ssd',
            'r8', 'r9', 'r10', 'r11', 
            'cr_ipsr', 'cr_iip', 'cr_ifs',
            'ar_unat', 'ar_pfs', 'ar_rsc', 'ar_rnat', 'ar_bspstore',
            'pr', 'b0', 'loadrs',
            'r1', 'r12',            
            'r13', 'ar_fpsr', 'r15', 'r14', 'r2', 'r3', 'r16', 'r17', 'r18', 'r19', 'r20', 'r21', 'r22', 'r23',
            'r24', 'r25', 'r26', 'r27',
            'r28', 'r29', 'r30', 'r31', 'ar_ccv',
            'f6a', 'f6b', 'f7a', 'f7b', 'f8a', 'f8b', 'f9a', 'f9b', 'f10a', 'f10b', 'f11a', 'f11b',
            ) 

        # from: $(XEN)/xen/include/public/arch-ia64.h
        self.xenfmt = '44Q6Q6Q'
        self.xenregs = self.elfregs + (
            'r4',    'r5',    'r6',    'r7',    'eml_unat',    'rfi_pfs',
            )



def read_objdump(text):
    '''read objdump -s output and return binary string'''

    lines = text.split('\n')
    dat = []
    for line in lines:
        m = re.search('^ [0-9a-f]+ (([0-9a-f]{8} ){1,4})', line)
        if m:
            nums = m.group(1)
            for m in re.finditer('[0-9a-f][0-9a-f]', nums):
                if m:
                    c = chr(int(m.group(0), 16))
                    dat.append(c)
    return ''.join(dat)


arch = {'x86_32': Register_x86_32,
        'x86_32pae': Register_x86_32,
        'x86_64': Register_x86_64,
        'ia64': Register_ia64,
        }

def Register(archname):
    return arch[archname]()


if __name__ == '__main__':
    import pprint
    # 'objdump -s -j .reg dumpfile'  output
    instr = '''dump:     file format elf32-i386

Contents of section .reg:
 0000 0022c8c0 00c0f4fb 01000000 00000000  ."..............
 0010 63000000 00000000 00000000 7b000000  c...........{...
 0020 7b000000 00000000 33000000 88ee5dc0  {.......3.....].
 0030 5af213c0 61000000 46020000 d8fed8c0  Z...a...F.......
 0040 69000000                             i...            
'''
    
    regstring = read_objdump(instr)
    reg = Register_x86_32()
    reg.fromElf(regstring)
    print 'ELF'
    pprint.pprint(reg.asElf())
    print 'Xen'
    pprint.pprint(reg.asXen())
    
    pprint.pprint(reg)
