#!/usr/bin/env python

'''this file provite page table class for dump file (x86_32 without PAE)'''

import struct


MASK_L1 = 0xffc00000L
MASK_L2 = 0x003ff000L
OFFSET_L1 = 0x003fffffL
OFFSET_L2 = 0x00000fffL
BASE_L1 = 0xffc00000L
BASE_L2 = 0xfffff000L
FLAG4MB = 0x00000080L
FLAGPRESENT = 0x00000001L

class PageTable:
    '''page table class for x86_32 without PAE'''
    def __init__(self, dump, mfn):
        '''create new pagetable from addressed page mfn'''
        self.dump = dump        # CoreDump obj
        self.arch = dump.arch
        self.pt = mfn           # mfn of l1 page table
        self.l1 = self.read_l1(self.pt)
        if self.arch.name != 'x86_32':
            raise NotImplementedError   # x86_32 only.

    def virt_to_machine(self, vaddr):
        '''return machine address from vaddr'''
        pte = self.l1[vaddr & MASK_L1]
        
        if pte & FLAG4MB:       # 4MB page
            return pte & BASE_L1 | vaddr & OFFSET_L1
        else:                   # 4kB page
            mfn = self.arch.maddr_to_mfn(pte)
            pt = self.read_l2(mfn)
            pte = pt[vaddr & MASK_L2]
            return pte & BASE_L2 | vaddr & OFFSET_L2

    v2m = virt_to_machine       # alias

    def read_l1(self, mfn):
        '''read level-1 pagetable'''

        return self.read_pt(mfn, 1)

    def read_l2(self, mfn):
        '''read level-2 pagetable'''
        
        return self.read_pt(mfn, 2)

    def read_pt(self, mfn, level):
        '''return pagetable'''

        pt = struct.unpack( "L" * 1024,  self.dump.read_page(mfn) )

        if level == 1:
            shift = 22
        elif level == 2:
            shift = 12

        phash = {}
        for (page, pte) in enumerate(pt):
            if pte & FLAGPRESENT:
                phash[long(page) << shift] = pte

        return phash

    def load_data(self, vbase, size):
        data = ''
        old_pfn = None
        for i in range(size):
            vaddr = vbase + i
            pfn = vaddr / self.arch.page_size
            if pfn != old_pfn:
                data += self.dump.read_page(self.arch.maddr_to_mfn(self.v2m(vaddr)))
            old_pfn = pfn

        offset = self.arch.page_offset(vbase)
        return data[offset:offset+size]

if __name__ == '__main__':
    import ElfCore
    xendump = ElfCore.ElfCoreReader('dump', 'x86_32')
    pt = PageTable(xendump, 0x176000 >> 12)
    print hex(pt.v2m(0xff177440L))    # dom0
