(* 
    Domains for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

let xc_handle = Eventchan.xc_interface_open ()

class domain (id : int) (connection : Connection.connection) =
object (self)
  val m_id = id
  val m_connection = connection
  val mutable m_input_list = []
  val mutable m_output_list = []
  val mutable m_dying = false
  val mutable m_shutdown = false
  method private connection = m_connection
  method private input_list = m_input_list
  method private output_list = m_output_list
  method add_input_message message = m_input_list <- m_input_list @ [ message ]
  method add_output_message message = m_output_list <- m_output_list @ [ message ]
  method can_read = self#connection#can_read
  method can_write = self#has_output_message && self#connection#can_write
  method destroy = self#connection#destroy
  method dying = m_dying <- true
  method has_input_message = List.length self#input_list > 0
  method has_output_message = List.length self#output_list > 0
  method id = m_id
  method input_message =
    let message = List.hd self#input_list in
    m_input_list <- List.tl m_input_list;
    message
  method input_messages = self#input_list
  method is_dying = m_dying
  method is_shutdown = m_shutdown
  method output_message =
    let message = List.hd self#output_list in
    m_output_list <- List.tl m_output_list;
    message
  method output_messages = self#output_list
  method read = match self#connection#read with Some (message) -> self#add_input_message message | None -> ()
  method shutdown = m_shutdown <- true
  method write = self#connection#write self#output_message
end

class domains =
object (self)
  val m_dominfo = Dominfo.init ()
  val m_entries = Hashtbl.create 8
  val mutable m_domains : domain list = []
  method private check domain =
    if Dominfo.info self#dominfo xc_handle domain#id = 1 && Dominfo.domid self#dominfo = domain#id
    then (
      if (Dominfo.crashed self#dominfo || Dominfo.shutdown self#dominfo) && not domain#is_shutdown then domain#shutdown;
      if Dominfo.dying self#dominfo then domain#dying
    );
    domain#is_dying || domain#is_shutdown
  method private dominfo = m_dominfo
  method private entries = m_entries
  method add domain =
    m_domains <- domain :: m_domains;
    Hashtbl.add self#entries domain#id 0
  method cleanup = List.fold_left (fun domains domain -> if self#check domain then domain :: domains else domains) [] self#domains
  method domains = m_domains
  method entry_count domain_id = Hashtbl.find self#entries domain_id
  method entry_decr domain_id =
    let entries = try pred (Hashtbl.find self#entries domain_id) with Not_found -> 0 in
    Hashtbl.replace self#entries domain_id (if entries < 0 then 0 else entries)
  method entry_incr domain_id = Hashtbl.replace self#entries domain_id (try succ (Hashtbl.find self#entries domain_id) with Not_found -> 1)
  method find_by_id domain_id = List.find (fun domain -> domain#id = domain_id) self#domains
  method remove (domain : domain) =
    m_domains <- List.filter (fun dom -> domain#id <> dom#id) self#domains;
    Hashtbl.remove self#entries domain#id;
    domain#destroy
  method timeout = if List.exists (fun domain -> domain#can_read || domain#can_write) self#domains then 0.0 else - 1.0
end

(* Initialise an unprivileged domain *)
let domu_init id remote_port mfn notify =
  let port = Eventchan.bind_interdomain id remote_port in
  let interface = new Xenbus.xenbus_interface port (Xenbus.map_foreign xc_handle id mfn) in
  let connection = new Connection.connection interface in
  if notify then Eventchan.notify port;
  new domain id connection

(* Check if a domain is unprivileged based on its ID *)
let is_unprivileged_id domain_id =
  domain_id > 0
  
(* Check if a domain is unprivileged *)
let is_unprivileged domain =
  is_unprivileged_id domain#id
