/**************************************************************************\
*//*! \file ef_vnic_svh.c VNIC semi-virtualised hardware layer

Copyright 2006 Solarflare Communications Inc,
               9501 Jeronimo Road, Suite 250,
               Irvine, CA 92618, USA

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License version 2 as published by the Free
Software Foundation, incorporated herein by reference.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*//*
\**************************************************************************/
#include <ci/driver/virtual/vnic.h>
#include <etherfabric/vi.h>
#include <etherfabric/tools.h>
#include "ef_vnic_netdev.h"

/* Find the SVH that will deal with the H/W we've been given. This
 * will all change when we pluginise (pluginify?) the SVH layer, but
 * we should probably ensure that the null SVH is built-in and thus
 * always present. */
static ef_vnic_svh_functions *ef_svh_locate_by_type(int type)
{
  ef_vnic_svh_functions *funcs;
  switch(type) {
    case MSG_HWTYPE_EF1:
      funcs = &ef_vnic_svh_ef1;
      break;
    case MSG_HWTYPE_NONE:
      funcs = &ef_vnic_svh_null;
      break;
    default:
      ci_log("Unsupported H/W type %d: defaulting to S/W only", type);
      funcs = &ef_vnic_svh_null;
      break;
  }
  return funcs;
}

void ef_vnic_svh_add_bufs(ef_vnic_svh *svh, struct ef_msg *msg)
{
  /* Pass it on. */
  svh->hw_fns->add_bufs(svh, msg);
}

enum ef_vnic_post_status ef_vnic_svh_tx_post(ef_vnic_svh *svh,
                                             ef_vnic_pktbuff *pktbuff)
{
  int rc, value, err, lookup, flags;

  /* Do slow path send if local mac lookup returns true, or fast path
     fails.  This rather assumes that the pktbuff is a sk_buff, and
     that the data field points to the dest mac address.
     
     Broadcasts will return a positive lookup as we've stuck that
     address in the hash table.  Multicast addresses should probably
     go in there too
  */
  spin_lock_irqsave(&svh->local_macs_lock, flags);
  lookup = ef_cuckoo_hash_lookup(&svh->local_macs_table, pktbuff->data, &value);
  spin_unlock_irqrestore(&svh->local_macs_lock, flags);

  if(!lookup) {
    rc = svh->hw_fns->tx_post(svh, pktbuff);
    if(rc != EF_VNIC_STATUS_FAIL)
      return rc;
  }

  err = ef_vnic_slowpath_tx(pktbuff, svh);
  rc = (err == 0) ? EF_VNIC_STATUS_GOOD : EF_VNIC_STATUS_FAIL;

  return rc;
}

void ef_vnic_svh_unpost_all(ef_vnic_svh *svh);

int ef_vnic_svh_poll(ef_vnic_svh *svh, int rx_packets)
{
  ef_vnic_pktbuff *pktbuff;
  int rx_done;
  ef_vnic_bec *bec = EF_VNIC_BEC_FROM_SVH(svh);

  rx_done = 0;
  while(sh_fifo2_num(&bec->rcv_pkts) > 2) {
    /* The first 2 bytes are the packet length, in the ONE TRUE endianness */
    __u8 lo, hi;
    unsigned  len, avail;
    /* Get the length */
    sh_fifo2_get(&bec->rcv_pkts, &lo);
    sh_fifo2_get(&bec->rcv_pkts, &hi);
    len = lo + (hi << 8);
    VNIC_VERB(ci_log("Got %d bytes from BEND.", len));

    /* Make sure the packet length is sane. */
    avail = sh_fifo2_num(&bec->rcv_pkts);
    if(len < ETH_ZLEN || len > ETH_FRAME_LEN || len > avail) {
      ci_log("%s: Packet length %d isn't sane with %d bytes available.",
             __FUNCTION__, len, avail);
      /* The FIFO must be corrupt.  Drop the rest of the FIFO on the
       * floor. */
      sh_fifo2_rd_adv(&bec->rcv_pkts, avail);
      break;
    }

    /* Get a skb, of the correct length */
    pktbuff = ef_vnic_osd_new_rx_buffer(EF_VNIC_OSD_FROM_BEC(bec), len);
    if (pktbuff != NULL) {
      int toend = sh_fifo2_contig_num(&bec->rcv_pkts);

      /* We can only read up to one page past the end of the fifo */
      ci_assert(toend > len || len - toend < PAGE_SIZE);
      memcpy(ef_vnic_pktbuff_get_header_data(pktbuff),
             sh_fifo2_peekp(&bec->rcv_pkts), len);
      sh_fifo2_rd_adv(&bec->rcv_pkts, len);

      /* Pass up the network stack */
      ef_vnic_osd_rx_complete(EF_VNIC_OSD_FROM_BEC(bec), pktbuff, len);

      rx_done++;
      if(rx_done >= rx_packets)
        return rx_done;
    } else {
      ci_log("%s: Failed to get buffer", __FUNCTION__);
      /* Advance the pointer anyway. */
      sh_fifo2_rd_adv(&bec->rcv_pkts, len);
    }
  }
  /* Collect any completions */
  rx_done += svh->hw_fns->poll(svh, rx_packets-rx_done);

  return rx_done;
}

void ef_vnic_svh_updated_mtu(ef_vnic_svh *svh)
{
  svh->hw_fns->updated_mtu(svh);
}

void ef_vnic_svh_disable_interrupts(ef_vnic_svh *svh)
{
  svh->hw_fns->disable_interrupts(svh);
}

ci_boolean_t ef_vnic_svh_enable_interrupts(ef_vnic_svh *svh)
{
  if(sh_fifo2_num(&EF_VNIC_BEC_FROM_SVH(svh)->rcv_pkts) > 2)
    return CI_FALSE;

  return svh->hw_fns->enable_interrupts(svh);
}

ci_boolean_t ef_vnic_svh_check_interrupts(ef_vnic_svh *svh)
{
  if(sh_fifo2_num(&EF_VNIC_BEC_FROM_SVH(svh)->rcv_pkts) > 2)
    return CI_FALSE;

  return svh->hw_fns->check_interrupts(svh);
}

void ef_vnic_svh_get_mac_addr(ef_vnic_svh *svh,
                              ci_uint8 *buffer)
{
  memcpy(buffer, svh->ep_state.mac, ETH_ALEN);
}

int ef_vnic_svh_ctor(ef_vnic_svh *svh, struct ef_msg_hw *hw_msg)
{
  svh->hw_fns = ef_svh_locate_by_type(hw_msg->type);
  /* We should always get a valid SVH back. */
  ci_assert(svh->hw_fns != NULL);
  return svh->hw_fns->ctor(svh, hw_msg);
}

void ef_vnic_svh_dtor(ef_vnic_svh *svh)
{
  svh->hw_fns->dtor(svh);
}
