import libelf

class Arch:
    def __init__(self, name, kw):
        self.name = name
        self.kw = kw
        self.page_size = 1 << self.page_shift
        self.pagemask = self.page_size - 1
        self.stack_size = self.page_size << self.stack_order

    def __getattr__(self, name):
        if name in self.kw:
            return self.kw[name]
        else:
            raise AttributeError

    def maddr_to_mfn(self, maddr):
        '''return mfn from maddr'''
        return maddr >> self.page_shift
        
    def mfn_to_maddr(self, mfn):
        '''return maddr from mfn'''
        return mfn << self.page_shift

    def pfn_to_vaddr(self, pfn):
        '''return vaddr from pfn'''
        return pfn << self.page_shift

    def page_offset(self, maddr):
        '''return offset in page'''
        return maddr & self.pagemask

    def round_pgup(self, size):
        '''return size round by pagesize'''
        return ((size + (self.page_size - 1)) / self.page_size) * self.page_size

    def is_xen_virt(self, pfn):
        '''return pfn is in xen virtual address space'''
        return self.hypervisor_virt_start <= pfn <= self.hypervisor_virt_end



archparam = {'x86_32':
             {'page_shift': 12,
              'ptr_format':'L',
              'pfn_format':'L',
              'max_vcpu': 32,
              'max_cpu': 32,
              'elf_format': 'ELF32',
              'elf_machine': libelf.EM_386,
              'hypervisor_virt_start': 0xFC000L,
              'hypervisor_virt_end': 0xFFFFFL,
              'hypervisor_page_offset': 0xFF000000L,
              'stack_order': 1,
              },
             'x86_32pae':
             {'page_shift': 12,
              'ptr_format':'L',
              'pfn_format':'L',
              'max_vcpu': 32,
              'max_cpu': 32,
              'elf_format': 'ELF32',    # gdb doesn't support elf64-i386
              'elf_machine': libelf.EM_386,
              'hypervisor_virt_start': 0xF5800L,
              'hypervisor_virt_end': 0xFFFFFL,
              'hypervisor_page_offset': 0xFF000000L,
              'stack_order': 1,
              },
             'x86_64':
             {'page_shift': 12,
              'ptr_format':'Q',
              'pfn_format':'Q',
              'max_vcpu': 32,
              'max_cpu': 32,            # XXXX can set larger?
              'elf_format': 'ELF64',
              'elf_machine': libelf.EM_X86_64,
              'hypervisor_virt_start': 0xFFFF800000000L,
              'hypervisor_virt_end': 0xFFFF880000000L,
              'hypervisor_page_offset': 0xFFFF830000000000L,
              'stack_order': 2,
              },
             'ia64':
             {'page_shift': 14,
              'ptr_format':'Q',
              'pfn_format':'Q',
              'max_vcpu': 64,
              'max_cpu': 64,
              'elf_format': 'ELF64',
              'elf_machine': libelf.EM_IA_64,
              'hypervisor_virt_start': 0x3c00000000000L,
              'hypervisor_virt_end': 0x3d00000000000L,
              'hypervisor_page_offset': 0xe000000000000000L,
              'stack_order': 0,         # XXXXX it doesn't work in IA64
              }}

arch = {'x86_32': Arch('x86_32', archparam['x86_32']),
        'x86_32pae': Arch('x86_32pae', archparam['x86_32pae']),
        'x86_64': Arch('x86_64', archparam['x86_64']),
        'ia64': Arch('ia64', archparam['ia64']),
        }
