(* 
    Permissions for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

type access =
  | NONE
  | READ
  | WRITE
  | BOTH

type t =
  {
    access : access;
    domain_id : int
  }

let make access domain_id =
  {
    access = access;
    domain_id = domain_id
  }

let permission_of_string string =
  {
    access =
      (match string.[0] with
        | 'n' -> NONE
        | 'r' -> READ
        | 'w' -> WRITE
        | 'b' -> BOTH
        | _ -> raise (Constants.Xs_error (Constants.EINVAL, "permission_of_string", string)));
    domain_id = int_of_string (String.sub string 1 (pred (String.length string)))
  }

let string_of_permission permission =
  let perm_str =
    match permission.access with
    | NONE -> "n"
    | READ -> "r"
    | WRITE -> "w"
    | BOTH -> "b" in
  perm_str ^ (string_of_int permission.domain_id)

let check_access access1 access2 =
  match access1 with
  | READ | WRITE -> access2 = access1 || access2 = BOTH
  | _ -> access2 = access1

let compare permission1 permission2 =
  permission1.access = permission2.access && permission1.domain_id = permission2.domain_id

let get_path path =
  Store.root_path ^ ".permissions" ^ (if path = Store.root_path then Constants.null_string else path)

class permissions =
object(self)
  method add (store : string Store.store) (path : string) (domain_id : int) =
    let domain_id = if domain_id < 0 then 0 else domain_id
    and parent_path = Store.parent_path path in
    if not (store#node_exists (get_path parent_path)) then self#add store parent_path domain_id;
    let parent_permissions = self#get store parent_path in
    let new_permissions = if domain_id = 0 then parent_permissions else make (List.hd parent_permissions).access domain_id :: List.tl parent_permissions in
    self#set (List.map string_of_permission new_permissions) store path
  method check (store : string Store.store) path access domain_id =
    let domain_id = if domain_id < 0 then 0 else domain_id
    and permissions = self#get store path in
    if domain_id = 0
    then true
    else
      let default_permission = List.hd permissions
      and actual_permissions = List.tl permissions in
      if default_permission.domain_id = domain_id
      then true
      else check_access access (try (List.find (fun perm -> perm.domain_id = domain_id) actual_permissions).access with Not_found -> default_permission.access)
  method get (store : string Store.store) (path : string) =
    let ppath = get_path path in
    match store#read_node ppath with
    | Store.Value permissions | Store.Hack (permissions, _) -> List.map permission_of_string (Utils.split permissions)
    | Store.Empty -> raise (Constants.Xs_error (Constants.EINVAL, "Permission.permissions#get", ppath))
    | Store.Children _ ->
        let parent_path = Store.parent_path path in
        let parent_permissions = self#get store parent_path in
        self#set (List.map string_of_permission parent_permissions) store path;
        parent_permissions
  method remove (store : string Store.store) path = store#remove_node (get_path path)
  method set (permissions : string list) (store : string Store.store) (path : string) =
    let ppath = get_path path in
    let parent_path = Store.parent_path path in
    if not (path = Store.root_path) && not (store#node_exists (get_path parent_path))
    then (
      let domain_id = (permission_of_string (List.hd permissions)).domain_id in
      self#add store parent_path domain_id
    );
    ignore (try store#read_node ppath with _ -> store#create_node ppath; store#read_node ppath);
    store#write_node ppath (Utils.combine_with_string permissions (String.make 1 Constants.null_char));
end
