(* 
    Utils for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

(* Print an error to standard output stream and die *)
let barf str =
  Printf.printf "FATAL: %s\n" str; flush stdout;
  ignore (exit 1)

(* Print an error to the error stream and die *)
let barf_perror str =
  Printf.eprintf "FATAL: %s\n" str; flush stderr;
  ignore (exit 1)

(* Convert a string of bytes into an int32 *)
let bytes_to_int32 bytes =
  let num_bytes = 4 in
  (* Convert bytes to an int32 *)
  let rec loop i n =
    if i >= num_bytes
    then n
    else loop (succ i) (Int32.add (Int32.shift_left n 8) (Int32.of_int (int_of_char bytes.[(num_bytes - 1) - i])))
  in
  loop 0 Int32.zero

(* Convert a string of bytes into an int *)
let bytes_to_int bytes =
  Int32.to_int (bytes_to_int32 bytes)

let combine lst =
  List.fold_left (fun rest i -> rest ^ i) Constants.null_string lst

let combine_with_string lst str =
  List.fold_left (fun rest i -> rest ^ i ^ str) Constants.null_string lst

(* Convert an int into a string of bytes *)
let int32_to_bytes num =
  let num_bytes = 4 in
  let bytes = String.create num_bytes in
  let rec loop i n =
    if i < num_bytes
    then (
      bytes.[i] <- char_of_int (Int32.to_int (Int32.logand 0xFFl n));
      loop (succ i) (Int32.shift_right_logical n 8)
    )
  in
  loop 0 num;
  bytes

(* Convert an int into a string of bytes *)
let int_to_bytes num =
  int32_to_bytes (Int32.of_int num)

(* Null terminate a string *)
let null_terminate str =
  str ^ (String.make 1 Constants.null_char)

(* Remove the last element from a list *)
let remove_last list =
  let length = pred (List.length list) in
  let rec loop n = (if (n = length) then [] else (List.nth list n :: loop (succ n))) in
  loop 0

(* Clean a string up for output *)
let sanitise_string str =
  let replacement_string = String.make 1 ' ' in
  let rec replace_nulls s =
    try
      let i = String.index s Constants.null_char in
      (String.sub s 0 i) ^ replacement_string ^ (replace_nulls (String.sub s (succ i) ((String.length s) - (succ i))))
    with Not_found -> s
  in
  replace_nulls str

(* Split a string into a list of strings based on the specified character *)
let split_on_char str char =
  let rec split_loop s =
    if (s = Constants.null_string) then []
    else
      try
        let null_index = String.index s char in
        String.sub s 0 null_index :: split_loop (String.sub s (succ null_index) ((String.length s) - (succ null_index)))
      with Not_found -> [ s ] | Invalid_argument _ -> []
  in
  split_loop str

(* Split a string into a list of strings based on the null character *)
let split str =
  split_on_char str Constants.null_char

(* Strip the trailing null byte off a string, if there is one *)
let strip_null str =
  if String.length str = 0 then str
  else
    let last = pred (String.length str) in
    if str.[last] = Constants.null_char then String.sub str 0 last else str

(* Return if a string contains another string *)
let rec strstr s1 s2 =
  try
    let i = String.index s1 s2.[0] in
    if String.length (String.sub s1 i ((String.length s1) - i)) < String.length s2
    then false
    else if String.sub s1 i (String.length s2) = s2
    then true
    else strstr (String.sub s1 (succ i) ((String.length s1) - (succ i))) s2
  with Not_found -> false
