/**************************************************************************\
*//*! \file ef_vnic_bufs.c 

Copyright 2006 Solarflare Communications Inc,
               9501 Jeronimo Road, Suite 250,
               Irvine, CA 92618, USA

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License version 2 as published by the Free
Software Foundation, incorporated herein by reference.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*//*
\**************************************************************************/

#include <linux/spinlock.h>
#include "etherfabric/vi.h"
#include "etherfabric/tools.h"
#include <ci/driver/virtual/vnic.h>
#include "ci/xen/ef_msg_iface.h"
#include "ef_vnic_bufs.h"


static int ef_vnic_alloc_buffer_mem(void ***page_list, int pages)
{
  int n;

  *page_list = kzalloc(pages * sizeof(struct page *), GFP_ATOMIC);
  if (*page_list == NULL) {
    return -ENOMEM;
  }

  for (n = 0; n < pages; n++) {
    void *tmp = (void*)__get_free_page(GFP_ATOMIC);
    if (tmp == NULL)
      break;

    (*page_list)[n] = tmp; /* virt_to_page(tmp) */
  }

  if(n != pages){
    ci_log("%s: not enough pages: %d != %d", __FUNCTION__, n, pages);
    for(; n >= 0; n--)
      free_page((unsigned long)((*page_list)[n]));
    kfree(*page_list);
    return -ENOMEM;
  }

  return 0;
}

/* Allocate memory for the buffer manager and create a lock */
struct ef_vnic_bufinfo *ef_vnic_init_bufs(void)
{
  struct ef_vnic_bufinfo *res = kzalloc(sizeof(*res), GFP_KERNEL);
  if (res) {
    spin_lock_init(&res->lock);
    res->first_free = -1;

    if(ef_vnic_alloc_buffer_mem(&res->page_list, EF_BUF_MAX_PAGES)){
      kfree(res);
      return NULL;
    }
  }
  return res;
}

extern void ef_vnic_fini_bufs(struct ef_vnic_bufinfo *bufs)
{
  int i;

  /* TODO do we need to do (can we do?) anything about the grant of
     these pages? */
  for(i = 0; i < EF_BUF_MAX_PAGES; i++){
    free_page((unsigned long)(bufs->page_list[i]));
    kfree(bufs->page_list);
  }

  kfree(bufs);
}



/* TODO need to do the opposite of some of this when we go away */
int ef_vnic_buf_map_request(struct xenbus_device *dev,
                            ef_vnic_bufinfo *manager, struct ef_msg *msg, 
                            int pages, int offset)
{
  int i, pfn, mfn;

  ef_msg_init(msg, EF_MSG_MAPBUF);

  ci_assert_le(pages, EF_MSG_MAX_PAGE_REQ);

  msg->u.mapbufs.pages = pages;

  for (i = 0; i < msg->u.mapbufs.pages; i++) {
    pfn = page_to_pfn(virt_to_page(manager->page_list[offset+i]));
    mfn = pfn_to_mfn(pfn);
    VNIC_VERB(ci_log("Granting page %d, mfn %08x", i, mfn));
    msg->u.mapbufs.reqs[i].grant = xenbus_grant_ring(dev, pfn_to_mfn(pfn));
    msg->u.mapbufs.mfns[i] = mfn;
    if (msg->u.mapbufs.reqs[i].grant < 0) {
      /* TODO could really do with undoing any grants we've already
         made in this loop */
      ci_log("Failed to grant buffer: %d\n", msg->u.mapbufs.reqs[i].grant);
      return -EIO;
    }
  }

  /* This is interpreted on return as the offset in the the page_list */
  msg->u.mapbufs.reqid = offset;

  return 0;
}

/* Fill out a request for some buffers */
int ef_vnic_buf_request(ef_vnic_bufinfo *manager, struct ef_msg *msg)
{
  ef_msg_init(msg, EF_MSG_GETBUF);
  ci_assert_equal(manager->npages, 0);
  msg->u.bufs.pages = EF_BUF_MAX_PAGES;
  ci_log("Will request %d pages.", msg->u.bufs.pages);
  ci_log("EF_BUF_PAGES_PER_BLOCK = %d", EF_BUF_PAGES_PER_BLOCK);
  ci_log("EF_BUF_BUFS_PER_BLOCK = %d", EF_BUF_BUFS_PER_BLOCK);
  ci_log("EF_BUF_BUFS_PER_PAGE = %d", EF_BUF_BUFS_PER_PAGE);
  ci_assert_equal(EF_BUF_BUFS_PER_BLOCK,
                  EF_BUF_BUFS_PER_PAGE * EF_BUF_PAGES_PER_BLOCK);
  return 0;
}

/* Process a response to a buffer request. */
int ef_vnic_add_bufs(struct ef_vnic_bufinfo *manager, struct ef_msg *msg)
{
  int rc = 0, msg_pages;
  union ef_page_req *msg_reqs;
  efab_buffer_addr_t msg_buf;
  unsigned flags;
  int i;
  int newtot;
  int old_block_count;
  int new_block_count;

  if(msg->id == (EF_MSG_GETBUF | EF_MSG_REPLY)){
    msg_pages = msg->u.bufs.pages;
    msg_reqs = msg->u.bufs.reqs;
    msg_buf = msg->u.bufs.buf;
  }
  else {
    ci_assert(msg->id == (EF_MSG_MAPBUF | EF_MSG_REPLY));
    msg_pages = msg->u.mapbufs.pages;
    msg_reqs = msg->u.mapbufs.reqs;
    msg_buf = msg->u.mapbufs.buf;
  }

  spin_lock_irqsave(&manager->lock, flags);
  newtot = manager->npages + msg_pages;
  old_block_count = ( (manager->npages + EF_BUF_PAGES_PER_BLOCK - 1) >>
                      EF_BUF_PAGES_PER_BLOCK_SHIFT );
  new_block_count = ( (newtot + EF_BUF_PAGES_PER_BLOCK - 1) >>
                      EF_BUF_PAGES_PER_BLOCK_SHIFT );
  ci_assert_le(newtot, EF_BUF_MAX_PAGES);
  for (i = old_block_count; i < new_block_count; i++) {
    struct ef_vnic_pkt_desc *block;
    ci_assert_lt(i, EF_BUF_NUM_BLOCKS);
    ci_log("Allocating block %d", i);
    if(manager->desc_blocks[i] != NULL) {
      ci_log("Not needed");
      continue;
    }
    block = kzalloc(EF_BUF_BUFS_PER_BLOCK * sizeof(ef_vnic_pkt_desc),
                    GFP_ATOMIC);
    if(block == NULL) {
      spin_unlock_irqrestore(&manager->lock, flags);
      return -ENOMEM;
    }
    manager->desc_blocks[i] = block;
  }
  for (i = manager->npages; i < newtot; i++) {
    int j = i - manager->npages;
    int block_num;
    int block_idx;
    struct ef_vnic_pkt_desc *pkt;

    block_num = i >> EF_BUF_PAGES_PER_BLOCK_SHIFT;
    block_idx = (2*i) & (EF_BUF_BUFS_PER_BLOCK-1);

    pkt = manager->desc_blocks[block_num] + block_idx;

    ci_assert_equal(EF_BUF_BUFS_PER_PAGE, 2);

    pkt[0].pkt_kva = msg_reqs[j].ptr;
    pkt[0].pkt_buff_addr = msg_buf + PAGE_SIZE/2 * (2 * j);
    pkt[0].u.next_free = manager->first_free;
    manager->first_free = 2*i;
    *(int*)(pkt[0].pkt_kva)=2*i;

    pkt[1].pkt_kva = msg_reqs[j].ptr + PAGE_SIZE/2;
    pkt[1].pkt_buff_addr = msg_buf + PAGE_SIZE/2 * (2 * j + 1);
    pkt[1].u.next_free = manager->first_free;
    manager->first_free = 2*i + 1;
    *(int*)(pkt[1].pkt_kva)=2*i+1;

    VNIC_VERB(ci_log("buf %d desc %p kva %p buffaddr %x",
                     2 * i,     pkt+0, pkt[0].pkt_kva, pkt[0].pkt_buff_addr));
    VNIC_VERB(ci_log("buf %d desc %p kva %p buffaddr %x",
                     2 * i + 1, pkt+1, pkt[1].pkt_kva, pkt[1].pkt_buff_addr));
  }
  manager->npages = newtot;
  spin_unlock_irqrestore(&manager->lock, flags);
  ci_log("Added %d pages. Total is now %d",
         msg_pages, manager->npages);
  return rc;
}

ef_vnic_pkt_desc *ef_vnic_buf_find(ef_vnic_bufinfo *manager, __u16 id)
{
  ef_vnic_pkt_desc *pkt;
  int block_num = id >> EF_BUF_BUFS_PER_BLOCK_SHIFT;
  int block_idx = id & (EF_BUF_BUFS_PER_BLOCK - 1);
  ci_assert_lt(id, manager->npages * EF_BUF_BUFS_PER_PAGE);
  ci_assert_lt(block_num, EF_BUF_NUM_BLOCKS);
  ci_assert_lt(block_idx, EF_BUF_BUFS_PER_BLOCK);
  pkt = manager->desc_blocks[block_num] + block_idx;
  return pkt;
}

/* Allocate a buffer from the buffer manager */
struct ef_vnic_pkt_desc *ef_vnic_buf_get(ef_vnic_bufinfo *manager, __u16 *id)
{
  int bufno = -1;
  unsigned flags;
  struct ef_vnic_pkt_desc *buf = NULL;
  /* Any spare? */
  if (manager->first_free == -1)
    return NULL;
  /* Take lock */
  spin_lock_irqsave(&manager->lock, flags);
  bufno = manager->first_free;
  if (bufno != -1) {
    buf = ef_vnic_buf_find(manager, bufno);
    manager->first_free = buf->u.next_free;
    /* Return number of buffer in id */
    *id = (__u16)bufno;
    manager->nused++;
  }
  /* Release lock */
  spin_unlock_irqrestore(&manager->lock, flags);
  /* Tell the world */
  VNIC_VERB(ci_log("Allocated buffer %i, buffaddr %x",
                   bufno, buf->pkt_buff_addr));
  return buf;
}

/* Release a buffer back to the buffer manager pool */
void ef_vnic_buf_put(ef_vnic_bufinfo *manager, __u16 id)
{
  int bufno = id;
  unsigned flags;
  struct ef_vnic_pkt_desc *buf = ef_vnic_buf_find(manager, id);
  VNIC_VERB(ci_log("Freeing buffer %i", bufno));
  ci_assert_nequal(id, (__u16)-1);
  spin_lock_irqsave(&manager->lock, flags);
  buf->u.next_free = manager->first_free;
  manager->first_free = bufno;
  manager->nused--;
  spin_unlock_irqrestore(&manager->lock, flags);
}

/* Access the context field in the descriptor of a buffer */
ef_vnic_pktbuff *ef_vnic_buf_pktbuff(ef_vnic_bufinfo *manager, __u16 id)
{
  return ef_vnic_buf_find(manager, id)->u.pkt_os_buff;
}

/*! \cidoxg_end */
