/**************************************************************************\
*//*! \file ef_msg_iface.c Frontend <-> backend message communication

Copyright 2006 Solarflare Communications Inc,
               9501 Jeronimo Road, Suite 250,
               Irvine, CA 92618, USA

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License version 2 as published by the Free
Software Foundation, incorporated herein by reference.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

 *//*
\**************************************************************************/

#include "ci/xen/ef_msg_iface.h"

#define EF_MSG_Q_SIZE (1024)
#define EF_MSG_Q_MASK (EF_MSG_Q_SIZE - 1)

#ifdef NDEBUG
#define EF_CHECK_SPAGE(_p)
#define EF_SHOW_QUEUE(_t, _q, _id)
#else
#define EF_CHECK_SPAGE(_p, _errval)\
  if (_p->magic != EF_MSG_MAGIC) {\
    printk(KERN_ERR "%s: passed invalid shared page %p!\n", __FUNCTION__, _p);\
    return _errval;\
}
#define EF_SHOW_QUEUE(_t, _q, _id)\
 printk(_t ": queue %d write %x read %x base %x limit %x\n", _id, _q->write\
 , _q->read, _q->base, _q->limit);
#endif

/* We've been passed at least 2 pages. 1 control page and 1 or more data pages. */
int ef_msg_init_page(void *mem, int len)
{
  struct ef_shared_page *shared_page = (struct ef_shared_page*)mem;


  /* We could be cleverer and not require this, by fiddling the
  * setting of base below.  shrug. */
  if ((__u32)shared_page & EF_MSG_Q_MASK)
    return -EINVAL;

  shared_page->magic = EF_MSG_MAGIC;

  shared_page->aflags = 0;

  return 0;
}

void ef_msg_init_queue(sh_msg_fifo2 *queue, struct ef_msg_queue *indices,
                       struct ef_msg *base, int size)
{
  queue->fifo = base;
  spin_lock_init(&queue->lock);
  sh_fifo2_init(queue, size-1, &indices->read, &indices->write);
}


static inline int _ef_msg_send(struct ef_shared_page *sp, sh_msg_fifo2 *queue,
                               struct ef_msg *msg)
{
  int rc = 0;
  EF_CHECK_SPAGE(sp, -EINVAL);
  rmb();
  if (sh_fifo2_not_full(queue)) {
      sh_fifo2_put(queue, *msg);
  } else {
    rc = -ENOSPC;
  }
  wmb();
  return rc;
}

/* Send a message on the specified FIFO. Returns 0 on success, -errno on
 * failure. The message in msg is copied to the current slot of the FIFO. */
int ef_msg_send(struct ef_shared_page *sp, sh_msg_fifo2 *q, 
                struct ef_msg *msg)
{
  unsigned flags;
  int rc;
  spin_lock_irqsave(&q->lock, flags);
  rc = _ef_msg_send(sp, q, msg);
  spin_unlock_irqrestore(&q->lock, flags);
  return rc;
}

/* As ef_msg_send but also posts a notification to the far end. */
int ef_msg_send_notify(struct ef_shared_page *sp, ef_notify_t ev, 
                       sh_msg_fifo2 *q, struct ef_msg *msg)
{
  unsigned flags;
  int rc;
  spin_lock_irqsave(&q->lock, flags);
  rc = _ef_msg_send(sp, q, msg);
  spin_unlock_irqrestore(&q->lock, flags);
  if (rc >= 0)
    ef_hyperop_remote_irq(ev);
  return rc;
}


/* As above, but doesn't try to take the lock - you must have
   previously locked the queue */
int ef_msg_send_locked(struct ef_shared_page *sp, sh_msg_fifo2 *q, 
                       struct ef_msg *msg)
{
  return _ef_msg_send(sp, q, msg);
}

/* As above, but doesn't try to take the lock - you must have
   previously locked the queue */
int ef_msg_send_notify_locked(struct ef_shared_page *sp, ef_notify_t ev, 
                              sh_msg_fifo2 *q, struct ef_msg *msg)
{
  int rc;
  rc = _ef_msg_send(sp, q, msg);
  if (rc >= 0)
    ef_hyperop_remote_irq(ev);
  return rc;
}


/* Look at a received message, if any, so a decision can be made about
   whether to read it now or not.  Cookie is a bit of debug which is
   set here and checked when passed to ef_msg_recv_next() */
int ef_msg_peek(struct ef_shared_page *sp, sh_msg_fifo2 *queue, 
                struct ef_msg *msg, int *cookie)
{
  unsigned flags;
  int rc = 0;
  EF_CHECK_SPAGE(sp, -EINVAL);
  spin_lock_irqsave(&queue->lock, flags);
  rmb();
  if (sh_fifo2_is_empty(queue)) {
    rc = -ENOENT;
  } else {
    *msg = sh_fifo2_peek(queue);
    *cookie = *(queue->fifo_rd_i);
  }
  spin_unlock_irqrestore(&queue->lock, flags);
  return rc;
}

/* Move the queue onto the next element, used after finished with a
   peeked msg */
int ef_msg_recv_next(struct ef_shared_page *sp, sh_msg_fifo2 *queue, int cookie)
{
  unsigned flags;
  EF_CHECK_SPAGE(sp, -EINVAL);
  spin_lock_irqsave(&queue->lock, flags);
  rmb();
  /* Mustn't be empty */
  ci_assert(!sh_fifo2_is_empty(queue));
  /* Check cookie matches, i.e. we're advancing over the same message
     as was got using peek */
  ci_assert_equal(cookie, *(queue->fifo_rd_i));
  sh_fifo2_rd_next(queue);
  wmb();
  spin_unlock_irqrestore(&queue->lock, flags);
  return 0;
}


/* Receive a message on the specified FIFO. Returns 0 on success, -errno on
 * failure. */
int ef_msg_recv(struct ef_shared_page *sp, sh_msg_fifo2 *queue, struct ef_msg *msg)
{
  unsigned flags;
  int rc = 0;
  EF_CHECK_SPAGE(sp, -EINVAL);
  spin_lock_irqsave(&queue->lock, flags);
  rmb();
  if (sh_fifo2_is_empty(queue)) {
    rc = -ENOENT;
  } else {
    sh_fifo2_get(queue, msg);
  }
  wmb();
  spin_unlock_irqrestore(&queue->lock, flags);
  return rc;
}

/* Start sending a message without copying. returns a pointer to a message
 * that will be filled out in place. The queue is locked until the message 
 * is sent. */
struct ef_msg *ef_msg_start_send(struct ef_shared_page *sp,
                                 sh_msg_fifo2 *queue, unsigned *flags)
{
  struct ef_msg *msg;
  EF_CHECK_SPAGE(sp, NULL);
  spin_lock_irqsave(&queue->lock, *flags);
  rmb();
  if (sh_fifo2_not_full(queue)) {
    msg = sh_fifo2_pokep(queue);
  } else {
    msg = NULL;
  }
  return msg;
}

static inline void _msg_complete(struct ef_shared_page *sp,
                                 sh_msg_fifo2 *queue,
                                 ef_lock_state_t *flags)
{
  sh_fifo2_wr_next(queue);
  wmb();
  spin_unlock_irqrestore(&queue->lock, *flags);
}

/* Complete the sending of a message started with ef_msg_start_send. The 
 * message is implicit since the queue was locked by _start */
void ef_msg_complete_send(struct ef_shared_page *sp, sh_msg_fifo2 *queue,
                           unsigned *flags)
{
  _msg_complete(sp, queue, flags);
}

/* As ef_msg_complete_send but does the notify. */
void ef_msg_complete_send_notify(struct ef_shared_page *sp, sh_msg_fifo2 *queue, 
                                 ef_lock_state_t *flags, ef_notify_t ev)
{
  _msg_complete(sp, queue, flags);
  ef_hyperop_remote_irq(ev);
}
