/**************************************************************************\
*//*! \file ef_filter.h Hardware filter insertion/removal for backend driver

Copyright 2006 Solarflare Communications Inc,
               9501 Jeronimo Road, Suite 250,
               Irvine, CA 92618, USA

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License version 2 as published by the Free
Software Foundation, incorporated herein by reference.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

 *//*
\**************************************************************************/

#ifndef EF_FILTER_H
#define EF_FILTER_H

#include <ci/driver/efab/internal.h>

/*! \remarks There are two different ways filters can be managed:
 * implicitly, as is done for unicast, where the packet is inspected and
 * a filter added if appropriate (possibly resulting in an existing
 * filter being evicted) or explicitly, as happens for multicast. Because
 * a filter can only forward to one queue, multicast needs to insert and
 * remove filters explicitly depending on the number of interfaces
 * listening on an address. The implicit case does not require the caller
 * to maintain any state  but for the explicit the caller must hold on to a
 * handle (the return from ef_bend_filter_add) which it can use for 
 * subsequent removal of that filter (passing it to ef_bend_filter_remove).
*/

/*! Filtering specification. This assumes that for VNIC support we will
 * always want wildcard entries, so only specifies the destination IP/port */
struct ef_filter_spec {
  void (*eviction_notice)(void *); /*!< Optional callback on eviction. */
  void *eviction_context; /*!< Context for eviction_notice */
  __u32 destip_be;    /*!< Destination IP in network order */
  __u16 destport_be;  /*!< Destination port in network order */
  __u8  proto;
};

/*! Allocate the filters for a bend and initialise associated variables.
 * \param bend : the ef_bend to initialise
 * \return 0 on success, or -errno 
 */
int ef_bend_filter_init(struct ef_bend *bend);

/*! Free the filters for a bend and mark as uninitialised.
 * \param bend : the ef_bend to shut down
 */
void ef_bend_filter_shutdown(struct ef_bend *bend);


/*! Add a filter for the specified IP/port to the backend. May evict an existing
 * filter to do this. Succeeds unless the spec. is not TCP or UDP
 * \param bend : the backend
 * \param filt : the filter specification to add
 * \return NULL on failure, else the handle to pass back in to delete the filter
*/
filter_resource_t * ef_bend_filter_check_add(struct ef_bend *bend, struct ef_filter_spec *filt);

/*! Remove a filter entry for the specific device and IP/port 
 * \param bend : the backend device the filter belongs to
 * \param fres : the handle returned by ef_bend_filter_add 
*/
void ef_bend_filter_remove(struct ef_bend *bend, filter_resource_t *fres);
#endif
