/**************************************************************************\
*//*! \file ef_bend_fwd.h Xen back end packet forwarding

Copyright 2006 Solarflare Communications Inc,
               9501 Jeronimo Road, Suite 250,
               Irvine, CA 92618, USA

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License version 2 as published by the Free
Software Foundation, incorporated herein by reference.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

 *//*
\**************************************************************************/
#ifndef EF_BEND_FWD_H
#define EF_BEND_FWD_H

/*! Init the forwarding infrastructure, sets up locks and registers
 * with driverlink 
 * \return 0 on success, or -ENOMEM if it couldn't get memory for the
 * forward table 
 */
extern int ef_bend_init_fwd(void);

/*! Shut down the forwarding and free memory. */
extern void ef_bend_shutdown_fwd(void);

/*! Add an entry to the forwarding table. 
 * \param mac : MAC address, used as hash key
 * \param ctxt : value to associate with key (can be NULL, see
 * ef_bend_fwd_set_context)
 * \return 0 on success, -ENOMEM if table was full and could no grow it
 */
extern int ef_bend_fwd_add(const __u8 *mac, void *ctxt);

/*! Set the context pointer for an existing fwd table entry.
 * \param mac : key that is already present in the table
 * \param context : new value to associate with key
 * \return 0 on success, -ENOENT if mac not present in table.
 */
extern int ef_bend_fwd_set_context(const __u8 *mac, void *context);

/*! Remove an entry from the forwarding table. 
 * \param mac : the MAC address to remove
 * \return nothing: it is not an error if the mac was not in the table */
extern void ef_bend_fwd_remove(const __u8 *mac);

/*! Get the hardware type for the NIC associated with a bend
 * \param bend : the backend in question
 * \return one of the MSG_HWTYPE* values defined in ef_msg_iface.h */
extern int ef_bend_hwtype(struct ef_bend *bend);

/*! Iterate the function provided over the forwarding table
 * 
 * NB. The forward table spinlock is held during the callback, so make
 * sure it doesn't recurse (or try to call other things that access
 * the forward table) and make it snappy so we don't get lots of
 * spinning waiting for the lock
 *
 * \param callback The function to call
 *  \param context An argument to pass to callback
 */
void ef_bend_for_each_bend(void (*callback)(struct ef_bend*, const void *), 
                           const void *context);

/*! \fn int forward_to_vnic(struct ef_bend *bend, struct sk_buff *skb )
 * \brief Send a packet through the RX slow path fifo to the VNIC 
 * 
 * Uses a spinlock internally to protect against concurrent access to
 * the fifo, but this spinlock is not necessarily used elsewhere to
 * protect the fifo, so that is left as an exercise to the coder.
 *
 * The lock is therefore only useful if the callers do not already
 * guarantee to not call this function concurrently.  At the moment
 * this is not the case only because the tx slow path can return
 * packets through here - all packets from driverlink are already
 * appropriately serialised.
 *
 * \param bend The backend driver data structure 
 * \param skb The packet to send, with MAC header stripped using skb_pull() 
 * \param have_lock : true if the caller holdsd the forwarding lock
 * \return 0 on success, or -ENOMEM if no space in the fifo
 **/
extern int forward_to_vnic(struct ef_bend *bend, struct sk_buff *skb, 
                           int have_lock );

/*! Handle a received packet: forward to appropriate vnics
 * \param skb The packet buffer
 * \return 1 if the packet has been claimed exclusively, else 0
*/
extern int ef_bend_rx_packet(struct sk_buff *skb);

/*! Signal a link state change
 * \param state The new state (true for up)
 */
extern void ef_bend_link_change(int state);

/*! Signal MTU change
 * \param mtu The new MTU
 */
extern void ef_bend_mtu_changed(int mtu);
#endif
