#!/usr/bin/env python
'''
dump file reader commandline interface.
'''

import sys
import struct

import XenCore
import ElfCore

from optparse import OptionParser

typemap = {
    'xen': XenCore.XenCoreReader,
    'elf': ElfCore.ElfCoreReader,
    }

def list2range(lst):
    '''make iterator of ranges of contiguous numbers from a list of integers'''

    tmplst = lst[:]
    tmplst.sort()
    ranges = []
    start = tmplst[0]

    currentrange = [start, start + 1]

    for item in tmplst[1:]:
        if currentrange[1] == item:
            # contiguous
            currentrange[1] += 1
        else:
            # new range start
            yield tuple(currentrange)
            currentrange = [item, item + 1]

    # last range
    yield tuple(currentrange)

def print_range(dump):
    '''print page frame numbers'''
    sys.stdout.write(str(list(list2range(dump.get_pagelist()))))
    sys.stdout.write('\n')

def print_page(dump, mfn):
    '''output raw page data'''
    content = dump.read_page(mfn)
    sys.stdout.write(content)

def main():
    '''commandline interface'''
    oparser = OptionParser()
    oparser.add_option("-p", "--mfn", dest="mfn", 
                       help="address as page number")
    oparser.add_option("-n", "--number", dest="pagenum", type="long",
                       help="address as page number")
    oparser.add_option("-a", "--addr", dest="addr", 
                       help="address")
    oparser.add_option("-r", "--print_range", dest="print_range", default=False,
                       action="store_true", help="print page number ranges")
    oparser.add_option("--arch", dest="arch", type="string", default="x86_32",
                       help="architecture ('ia64', 'x86_32', or 'x86_64')")
    oparser.add_option("-f", "--file", dest="filename", default="dump", 
                       type="string", help="dump file name")

    oparser.add_option("-t", "--type", dest="filetype", default="elf",
                       type="string", help='dump file type("xen" or "elf")')

    (options, args) = oparser.parse_args()

    # filetype, file, arch
    coretype = typemap[options.filetype]
    dump = coretype(options.filename, options.arch)

    # pagenum
    pages = 1
    if options.pagenum:
        pages = int(options.pagenum)

    # mfn
    mfn = None
    if options.mfn:
        try:
            mfn = int(options.mfn, 16)
        except:
            oparser.error("page number needs be hex")

    # address
    if options.addr:
        try:
            address = int(options.addr, 16)
            mfn = dump.arch.maddr_to_mfn(address)
        except StandardError, e:
            oparser.error("address needs be hex")
            raise e

    # output page or range
    if mfn != None:
        for i in range(pages):
            print_page(dump, mfn + i)
    elif options.print_range:
        print_range(dump)
    else:
        oparser.print_help()
        sys.exit()

if __name__ == '__main__':
    main()
