'''
Xen core file image interface.
This module define XenCoreReader, XenCoreWriter,
and their abstruct class XenCore
'''

import CoreDump
import struct

CORE_MAGIC = 0xF00FEBEDL
HEADER_FORMAT = "LLLLLL"

        
class XenCore(CoreDump.CoreDump):
    '''xen dump file abstract class'''
    def __init__(self, corefilename, arch):
        CoreDump.CoreDump.__init__(self, corefilename, arch)

        self.xch_magic = CORE_MAGIC     # magic number
        self.xch_nr_vcpus = 0           # numver of virtual cpus
        self.xch_nr_pages = 0           # number of pages
        self.xch_ctxt_offset = None     # context infos offset
        self.xch_index_offset = None    # pfn index offset
        self.xch_pages_offset = None    # page data offset

        self.ctxts = []                  # CPU contexts(as list of context obj)
        self.pages = []               # pfn index (as list of numbers)
        self.pageoffsetmap = {}         # mfn -> file offset hash
        self.file = None                # dump file itself

    ## CoreDump interface
    def read_page(self, mfn):
        '''return a page data.'''

        self.file.seek(self.pageoffsetmap[mfn])
        content = self.file.read(self.arch.page_size)
        return content

    def has_page(self, mfn):
        '''return a page is there or not'''
        return self.pageoffsetmap.has_key(mfn)

    def get_pagelist(self):
        '''return a list of all available mfn'''
        return self.pageoffsetmap.keys()

    ## pageoffsetmap maintainer
    def _build_pageoffsetmap(self):
        '''build self.pageoffsetmap from self.pages'''
        self.pageoffsetmap = {}

        if not self.pages:                   # no pages
            return

        for index, mfn in enumerate(self.pages):
            self.pageoffsetmap[mfn] = (self.xch_pages_offset +
                                       index * self.arch.page_size)

    def _update_offsets(self):
        '''recalc header offset data.'''

        self.xch_ctxt_offset = struct.calcsize(HEADER_FORMAT)
        self.xch_index_offset = (struct.calcsize(HEADER_FORMAT) +
                len(''.join([str(ctxt) for ctxt in self.ctxts])))
        self.xch_pages_offset = self.arch.round_pgup(
            self.xch_index_offset +
            self.xch_nr_pages * struct.calcsize(self.arch.pfn_format))

        self._build_pageoffsetmap()

    def clear_pages(self):
        '''delete all pages'''
        self.pages = []
        self.pageoffsetmap = {}


    def set_context(self, ctxts):
        '''set CPU(s) num and contexts'''
        self.xch_nr_vcpus = len(ctxts)
        self.ctxts = ctxts

        self._update_offsets()

    def set_registers(self, regs):
        '''overwrite user registers on context'''

        for i, reg in enumerate(regs):
            self.ctxts[i].set_register(reg)
            
    def set_ctrlreg(self, vcpu, num, val):
        '''set all ctrlreg[num] as val'''
        # XXXX it works on uni processor only
        # walkaround to generate Xen core header..
        self.ctxts[vcpu].set_ctrlreg(num, val)

    def set_v2m(self, v2m):
        # not used in Xen core format.
        pass
    
    def set_p2m(self, p2m):
        '''set pfn2mfn for this dump image'''
        # xen core format doesn't use p2m mapping,
        # it is processed by gdbserver-xen with cr3 value
        self.pages = p2m.values()
        self.pages.sort()
        self.xch_nr_pages = len(self.pages)

        self._update_offsets()
        


class XenCoreReader(XenCore):
    '''
    Xen core image file reader class.
    '''
    
    def __init__(self, corefilename, arch):
        XenCore.__init__(self, corefilename, arch)
        self.file = file(corefilename, 'r')
        self._readheader()

    def _readheader(self):
        '''read dumpfile header items'''

        self.file.seek(0)
        header = self.file.read(struct.calcsize(HEADER_FORMAT))
        (self.xch_magic,
         self.xch_nr_vcpus,
         self.xch_nr_pages,
         self.xch_ctxt_offset,
         self.xch_index_offset,
         self.xch_pages_offset) = struct.unpack(HEADER_FORMAT, header)

        if self.xch_magic != CORE_MAGIC:
            raise ValueError, 'magic number is not valid in Xen core image format'

        self.file.seek(self.xch_ctxt_offset)
        self.ctxts = self.file.read(self.xch_index_offset-self.xch_ctxt_offset)

        pt_format = self.arch.pfn_format * self.xch_nr_pages
        self.file.seek(self.xch_index_offset)
        pagetable = self.file.read(struct.calcsize(pt_format))
        self.pages = list(struct.unpack(pt_format, pagetable))
        self._build_pageoffsetmap()


class XenCoreWriter(XenCore):
    '''
    Xen core image file writer class.
    '''
    def __init__(self, corefilename, arch):
        XenCore.__init__(self, corefilename, arch)
        self.file = file(corefilename, 'r+')

    def write_header(self):
        '''write header'''
        self._update_offsets()

        header = struct.pack(
            HEADER_FORMAT,
            self.xch_magic, self.xch_nr_vcpus, self.xch_nr_pages,
            self.xch_ctxt_offset, self.xch_index_offset,
            self.xch_pages_offset)
        self.file.seek(0)
        self.file.write(header)

        self.file.seek(self.xch_ctxt_offset)
        self.file.write(''.join([str(ctxt) for ctxt in self.ctxts]))

        self.file.seek(self.xch_index_offset)
        pageindex = struct.pack(self.arch.pfn_format * len(self.pages), *self.pages)
        self.file.write(pageindex)

    def fetch_pages(self, other):
        '''copy pages from other dump'''
        count = 0
        pagebuf = {}
        pagelist = self.pages[:]
        pagelist.sort()
        for mfn in pagelist:
            pagebuf[mfn] = other.read_page(mfn)
            count += 1
            if count % 1024 == 0 or count == len(pagelist):
                for mfn in pagebuf:
                    self.write_page(mfn, pagebuf[mfn]) # write mfn
                pagebuf = {}

    def write_page(self, mfn, data):
        '''put a page data into file. header is not updated by this.'''
        if mfn in self.pageoffsetmap:
            # data on disk
            self.file.seek(self.pageoffsetmap[mfn])
            self.file.write(data)


